/*
 * Copyright (c) 1990 AT&T All Rights Reserved
 *	This is unpublished proprietary source code of AT&T
 *	The copyright notice above does not evidence any
 *	actual or intended publication of such source code.
 *
 * Pi Unix driver for Sun 4 (Sparc based) Workstations
 *	This file uses ptrace to examine running processes.
 *
 * D. A. Kapilow	12/14/90
 */
#include <stdio.h>
#include <signal.h>
#include <a.out.h>
#include <setjmp.h>
#include <malloc.h>
#include <sys/ptrace.h>
#include <sys/wait.h>
#include <machine/reg.h>
#include <sys/user.h>
#include <machine/vmparam.h>
#include <sys/core.h>
#include "dbmonitor/dbhost.h"

#define	bit(x)		(1<<(x-1))
#define	STEPWAIT	15
#define	SPARCBPT	0x91D02001		/* t 1 */
#define	SPARC_TRAPMASK	0xDFF82000
#define	SPARC_TRAP	0x91D02000
#define REGSIZE		(36 * 4)
#define	ISCORE(p)	((p)->pid < -1)

#define WAIT_POLL	0x1
#define	WAIT_PCFIX	0x2
#define	WAIT_DISCARD	0x4

/*
 * Data structure private to each process or core dump
 */
struct localproc {
	int		pid;
	int		opencnt;
	long		sigmsk;
	unsigned long	startdata;
	struct	Unixstate	state;
	struct localproc	*next;
	/* For core dumps */
	int		corefd;		/* Data, stack, regs */
	int		stabfd;		/* text */
	unsigned long	starttext;
	unsigned long	endtext;
	unsigned long	enddata;
	unsigned long	startstack;
	unsigned long	endstack;
	unsigned long	textoffset;
	unsigned long	dataoffset;
	unsigned long	stackoffset;
	unsigned long	regoffset;
};

extern int errno;
static struct localproc *phead;
static char procbuffer[128];
static int hangpid;
long pi_corepid = -1;
static unsigned long regaddr = -REGSIZE - 4;
static long bkpt_instr = SPARCBPT;
static char *pscmds[] = {
	"/bin/ps   ",
	"/bin/ps a ",
	"/bin/ps x ",
	"/bin/ps ax",
	0
};

/* Routines local to this file */
static int procwaitstop();
static int procwait();
static char *regrw();
static int doptrace();
static void coreclose();
static char *corerw();

/* Used before defined */
struct localproc *pi_pidtoproc();
char *pi_run();
char *pi_stop();

char	*pi_osname()		{ return "SunOS 4.1.1 Sparc"; }
char	**pi_getpscmds()	{ return pscmds; }
int	pi_getpsfield()		{ return 14; }
long	pi_regaddr()		{ return (long)regaddr; }
long	pi_scratchaddr()	{ return PAGSIZ; }
int	pi_nsig()		{ return NSIG; }
int	pi_exechangsupported()	{ return 0; }
int	pi_stabfdsupported()	{ return 0; }
int	pi_getsymtabfd()	{ return -1; }
int	pi_getppid()		{ return 0; }	/* Fix Me */

long pi_sigmaskinit()
{
	return	bit(SIGILL)|bit(SIGINT)|bit(SIGTRAP)|bit(SIGIOT)|
		bit(SIGEMT)|bit(SIGFPE)|bit(SIGBUS)|bit(SIGSEGV)|
		bit(SIGSYS)|bit(SIGPIPE)|bit(SIGSTOP);
}

struct localproc *pi_open(pid)
{
	register struct localproc *p = phead;
	struct exec exec;

	if (pid < 0)
		return 0;
	if (p = pi_pidtoproc(pid)) {
		p->opencnt++;
		return p;
	}
	p = (struct localproc *)malloc(sizeof(struct localproc));
	if (!p)
		return 0;
	kill(pid, SIGCONT);
	if (ptrace(PTRACE_ATTACH, pid, 0, 0, 0)) {
		free(p);
		return 0;
	}
	p->sigmsk = pi_sigmaskinit();
	p->pid = pid;
	p->opencnt = 1;
	p->next = phead;
	phead = p;
	procwait(p, 0);
	doptrace(p, PTRACE_READTEXT, pi_scratchaddr(),
		sizeof(exec), (int)&exec);
	p->startdata = N_DATADDR(exec);
	return p;
}

void pi_close(p)
struct localproc *p;
{
	register struct localproc *q;

	if (--p->opencnt != 0)
		return;
	if (ISCORE(p))
		coreclose(p);
	else if (p->state.state == UNIX_ACTIVE) {
		pi_stop(p);
		procwaitstop(p, WAIT_PCFIX);
		ptrace(PTRACE_DETACH, p->pid, 1, 0, 0);
	}
	else if (p->state.state != UNIX_ERRORED) {
		kill(p->pid, SIGSTOP);
		ptrace(PTRACE_DETACH, p->pid, 1, 0, 0);
	}
	if (p == phead)
		phead = p->next;
	else {
		for (q = phead; q->next != p; q = q->next)
			;
		q->next = p->next;
	}
	free(p);
}

struct localproc *pi_pidtoproc(pid)
{
	register struct localproc *p;

	for (p = phead; p; p = p->next)
		if (p->pid == pid)
			return p;
	return 0;
}

static jmp_buf saveenv;

static catchalarm()
{
	longjmp(saveenv, 1);
}

static int procwaitstop(p, flag)
struct localproc *p;
{
	int wret;
	int oldalarm;
	void (*oldsig)() = signal(SIGALRM, catchalarm);

	oldalarm = alarm(STEPWAIT);
	if (setjmp(saveenv)) {
		alarm(oldalarm);
		signal(SIGALRM, oldsig);
		kill(p->pid, SIGSTOP);
		return 1;
	}
	wret = procwait(p, flag);
	alarm(oldalarm);
	signal(SIGALRM, oldsig);
	if (!wret || p->state.state != UNIX_HALTED)
		return 1;
	return 0;
}

static int procwait(p, flag)
struct localproc *p;
{
	union wait tstat;
	int cursig;

again:
	if (p->pid != wait4(p->pid, &tstat,
			    (flag & WAIT_POLL) ? WNOHANG : 0, 0))
		return 0;
	if (flag & WAIT_DISCARD)
		return 1;
	if (WIFSTOPPED(tstat)) {
		cursig = tstat.w_stopsig;
		if (cursig == SIGSTOP)
			p->state.state = UNIX_HALTED;
		else if (cursig == SIGTRAP)
			p->state.state = UNIX_BREAKED;
		else {
			if (p->state.state == UNIX_ACTIVE &&
			    !(p->sigmsk&bit(cursig))) {
				ptrace(PTRACE_CONT, p->pid, 1, cursig, 0);
				goto again;
			}
			else {
				p->state.state = UNIX_PENDING;
				p->state.code = cursig;
			}
		}
	} else {
		p->state.state = UNIX_ERRORED;
		p->state.code = tstat.w_status & 0xFFFF;
	}
	return 1;
}

char *pi_run(p)
struct localproc *p;
{
	int cursig = 0;

	if (p->state.state == UNIX_PENDING)
		cursig = p->state.code;
	ptrace(PTRACE_CONT, p->pid, 1, cursig, 0);
	p->state.state = UNIX_ACTIVE;
	return 0;
}

char *pi_stop(p)
struct localproc *p;
{
	kill(p->pid, SIGSTOP);
	return 0;
}

char *pi_waitstop(p)
struct localproc *p;
{
	procwaitstop(p, WAIT_PCFIX);
	if (p->state.state == UNIX_BREAKED)
		return 0;
	else
		return "timeout waiting for breakpoint";
}

char *pi_destroy(p)
struct localproc *p;
{
	if (p->state.state == UNIX_ERRORED)
		return "already dead";
	if (p->state.state == UNIX_ACTIVE) {
		kill(p->pid, SIGSTOP);
		procwaitstop(p, WAIT_PCFIX);
	}
	ptrace(PTRACE_KILL, p->pid, 0, 0, 0);
	procwait(p, WAIT_DISCARD|WAIT_POLL);
	p->state.state = UNIX_ERRORED;
	p->state.code = SIGKILL;
	return 0;
}

void pi_getstate(p, s)
struct localproc *p;
struct Unixstate *s;
{
	if (p->state.state == UNIX_ACTIVE)
		procwait(p, WAIT_POLL|WAIT_PCFIX);
	*s = p->state;
}

char *pi_readwrite(p, buf, addr, r, w)
struct localproc *p;
char *buf;
unsigned long addr;
{
	char *err = 0;
	int wasactive = 0;
	enum ptracereq cmd;
	int off;

	if (ISCORE(p))
		return corerw(p, buf, addr, r, w);
	if (p->state.state == UNIX_ACTIVE) {
		kill(p->pid, SIGSTOP);
		procwaitstop(p, WAIT_PCFIX);
		if (p->state.state != UNIX_HALTED) {
			ptrace(PTRACE_CONT, p->pid, 1, 0, 0);
			procwaitstop(p, WAIT_DISCARD);
		} else
			wasactive = 1;
	}
	if (addr >= regaddr)
		err = regrw(p, buf, (int)(addr - regaddr), r, w);
	else {
		if (addr < p->startdata) {
			if (w)
				cmd = PTRACE_WRITETEXT;
			else {
				cmd = PTRACE_READTEXT;
				/* SunOS panic: Memory alignment error */
				if (off = addr % 4) {
				 if (r <= 8) {
				  char t[12];
				  addr -= off;
				  if (doptrace(p, cmd, (int)addr, sizeof(t), t))
					err = "ptrace failed";
				  else
					bcopy(&t[off], buf, r);
				 } else
					err = "Unimplemented: SunOS kernel bug";
				 if (wasactive)
					pi_run(p);
				 return err;
				}
			}
		} else {
			if (w)
				cmd = PTRACE_WRITEDATA;
			else
				cmd = PTRACE_READDATA;
		}
		if (doptrace(p, cmd, (int)addr, w ? w : r, buf))
			err = "ptrace failed";
	}
	if (wasactive)
		pi_run(p);
	return err;
}

static char *regrw(p, buf, offset, r, w)
struct localproc *p;
char *buf;
{
	struct regs rg;

	if (offset == (19 * 4))	{ /* %g0 */
		if (r)
			bzero(buf, 4);
		return 0;
	}
	if (doptrace(p, PTRACE_GETREGS, (int)&rg, 0, 0))
		return "PTRACE_GETREGS failed";
	if (w) {
		bcopy(buf, (char*)&rg + offset, w);
		if (doptrace(p, PTRACE_SETREGS, (int)&rg, 0, 0))
			return "PTRACE_SETREGS failed";
	} else
		bcopy((char*)&rg + offset, buf, r);
	return 0;
}

char *pi_setbkpt(p, addr)
struct localproc *p;
long addr;
{
	return pi_readwrite(p,(char*)&bkpt_instr, addr, 0, sizeof(bkpt_instr));
}

char *pi_step(p)
struct localproc *p;
{
	long save;
	long npc;

	/* Read value of register %npc */
	pi_readwrite(p, (char *)&npc, regaddr + nPC * 4, sizeof(npc), 0);
	/* Save what is at %npc */
	pi_readwrite(p, (char *)&save, npc, sizeof(save), 0);
	/* Set a breakpoint there */
	if (pi_setbkpt(p, npc))
		return "can't set breakpoint";
	/* Step it */
	doptrace(p, PTRACE_CONT, 1, 0, 0);
	procwaitstop(p, WAIT_PCFIX);
	/* Restore original instruction */
	pi_readwrite(p, (char *)&save, npc, 0, sizeof(save));
	return 0;
}

char *pi_ssig(p, sig)
struct localproc *p;
long sig;
{
	kill(p->pid, sig);
	return 0;
}

char *pi_csig(p)
struct localproc *p;
{
	if (p->state.state == UNIX_PENDING)
		p->state.state = UNIX_HALTED;
	return 0;
}

char *pi_sigmask(p, mask)
struct localproc *p;
long mask;
{
	p->sigmsk = mask;
	return 0;
}

static int doptrace(p, cmd, a, d, a2)
struct localproc *p;
enum ptracereq cmd;
{
	int ret;

	errno = 0;
	ret = ptrace(cmd, p->pid, (char *)a, d, (char *)a2);
	switch (cmd) {
		case PTRACE_CONT:
		case PTRACE_KILL:
		case PTRACE_SINGLESTEP:
		case PTRACE_ATTACH:
		case PTRACE_DETACH:
			break;
		default:
			if (!errno)
				procwait(p, WAIT_DISCARD);
			break;
	}
	return ret;
}

char *pi_signalname(sig)
long sig;
{
	char *cp;

	switch (sig) {
	case SIGHUP:	cp = "hangup"; break;
	case SIGINT:	cp = "interrupt"; break;
	case SIGQUIT:	cp = "quit"; break;
	case SIGILL:	cp = "illegal instruction"; break;
	case SIGTRAP:	cp = "trace/BPT"; break;
	case SIGIOT:	cp = "IOT instruction"; break;
	case SIGEMT:	cp = "EMT instruction"; break;
	case SIGFPE:	cp = "floating exception"; break;
	case SIGKILL:	cp = "kill"; break;
	case SIGBUS:	cp = "bus error"; break;
	case SIGSEGV:	cp = "memory fault"; break;
	case SIGSYS:	cp = "bad system call"; break;
	case SIGPIPE:	cp = "broken pipe"; break;
	case SIGALRM:	cp = "alarm call"; break;
	case SIGTERM:	cp = "terminated"; break;
	case SIGURG:	cp = "urgent condition"; break;
	case SIGSTOP:	cp = "stop"; break;
	case SIGCONT:	cp = "continue"; break;
	case SIGTSTP:	cp = "stop tty"; break;
	case SIGCHLD:	cp = "child termination"; break;
	case SIGTTIN:	cp = "stop tty input"; break;
	case SIGTTOU:	cp = "stop tty output"; break;
	case SIGIO:	cp = "i/o possible"; break;
	case SIGXCPU:	cp = "cpu timelimit"; break;
	case SIGXFSZ:	cp = "file sizelimit"; break;
	case SIGVTALRM:	cp = "virtual alarm"; break;
	case SIGPROF:	cp = "profiling alarm"; break;
	case SIGWINCH:	cp = "window changed"; break;
	default:
		cp = procbuffer;
		sprintf(procbuffer, "signal %d", sig);
		break;
	}
	return cp;
}

char *pi_proctime(p)
struct localproc *p;
{
	int offset, size;
	int wasactive = 0;
	long *lp;
	struct user *u = 0;
	struct rusage ru;

	offset = (char*)&u->u_ru - (char *)u;
	size = 2 * sizeof(struct timeval);

	if (p->state.state == UNIX_ACTIVE) {
		kill(p->pid, SIGSTOP);
		procwaitstop(p, WAIT_PCFIX);
		if (p->state.state != UNIX_HALTED) {
			ptrace(PTRACE_CONT, p->pid, 1, 0, 0);
			procwaitstop(p, WAIT_DISCARD);
		} else
			wasactive = 1;
	}
	lp = (long *)&ru.ru_utime;
	while(size) {
		errno = 0;
		*lp = ptrace(PTRACE_PEEKUSER, p->pid, offset, 0, 0);
		if (errno)
			break;
		procwait(p, WAIT_DISCARD);
		offset += 4;
		size -= 4;
		lp++;
	}
	if (wasactive)
		pi_run(p);
	if (errno)
		return "";
	sprintf(procbuffer, "%d.%du %d.%ds",
		ru.ru_utime.tv_sec,
		ru.ru_utime.tv_usec/100000,
		ru.ru_stime.tv_sec,
		ru.ru_stime.tv_usec/100000);
	return procbuffer;
}

int pi_atsyscall(p)
struct localproc *p;
{
	long pc, instr; 

	pi_readwrite(p, (char *)&pc, regaddr + PC * 4, sizeof(pc), 0);
	pi_readwrite(p, (char *)&instr, pc, sizeof(instr), 0);
	return ((instr & SPARC_TRAPMASK) == SPARC_TRAP);
}

char *pi_exechang(p, e)
struct localproc *p;
long e;
{
	return "not supported";
}

/*
 * This function and the function below are for starting new
 * processes from pi.
 */
int pi_hang(cmd)
char *cmd;
{
	char *argv[10], *cp;
	int i;
	struct localproc *p;
	struct exec exec;
	
	i = strlen(cmd);
	if (++i > sizeof(procbuffer)) {
		i = sizeof(procbuffer) - 1;
		procbuffer[i] = 0;
	}
	bcopy(cmd, procbuffer, i);
	argv[0] = cp = procbuffer;
	for(i = 1;;) {
		while(*cp && *cp != ' ')
			cp++;
		if (!*cp) {
			argv[i] = 0;
			break;
		} else {
			*cp++ = 0;
			while (*cp == ' ')
				cp++;
			if (*cp)
				argv[i++] = cp;
		}
	}
	hangpid = fork();
	if (!hangpid){
		int fd;
		for( fd = 0; fd < NOFILE; ++fd )
			close(fd);
		open("/dev/null", 2);
		dup2(0, 1);
		dup2(0, 2);
		setpgrp(0, getpid());
                ptrace(PTRACE_TRACEME, 0, 0, 0, 0);
                execvp(argv[0], argv);
		exit(0);
	}
	if (hangpid < 0)
		return 0;
	p = (struct localproc *)malloc(sizeof(struct localproc));
	if (!p) {
		kill(9, hangpid);
		return 0;
	}
	p->sigmsk = pi_sigmaskinit();
	p->pid = hangpid;
	if (!procwait(p, 0)) {
		free(p);
		return 0;
	}
	if (p->state.state == UNIX_BREAKED)
		p->state.state = UNIX_HALTED;
	doptrace(p, PTRACE_READTEXT, pi_scratchaddr(),
		sizeof(exec), (int)&exec);
	p->startdata = N_DATADDR(exec);
	p->opencnt = 0;
	p->next = phead;
	phead = p;
	return hangpid;
}

/*
 * The functions below are for debugging core dumps
 */
struct localproc *pi_coreopen(corep, symtabp)
char *corep, *symtabp;
{
	register struct localproc *p;
	int symtabfd, corefd = -1;
	int mode;
	struct core c;

	if ((symtabfd = open(symtabp, 0)) < 0)
		return 0;
	for (mode = 2; corefd < 0 && mode >= 0; mode--)
		corefd = open(corep, mode);
	if (corefd < 0) {
		close(symtabfd);
		return 0;
	}
	if (read(corefd, (char*)&c, sizeof(c)) != sizeof(c) ||
	    c.c_magic != CORE_MAGIC ||
	    c.c_aouthdr.a_machtype != M_SPARC ||
	    !(p = (struct localproc *)malloc(sizeof(struct localproc)))) {
		close(corefd);
		close(symtabfd);
		return 0;
	}
	p->pid = --pi_corepid;
	p->opencnt = 1;
	p->next = phead;
	phead = p;
	p->corefd = corefd;
	p->stabfd = symtabfd;
	p->starttext = N_TXTADDR(c.c_aouthdr);
	p->endtext = p->starttext + c.c_tsize;
	p->startdata = N_DATADDR(c.c_aouthdr);
	p->enddata = p->startdata + c.c_dsize;
	p->endstack = USRSTACK;
	p->startstack = p->endstack - c.c_ssize;
	p->textoffset = N_TXTOFF(c.c_aouthdr);
	p->dataoffset = c.c_len;
	p->stackoffset = p->dataoffset + c.c_dsize;
	p->regoffset = (char *)&c.c_regs - (char*)&c;
	p->state.state = UNIX_PENDING;
	p->state.code = c.c_signo;
	return p;
}

static void coreclose(p)
struct localproc *p;
{
	close(p->corefd);
	close(p->stabfd);
}

static char *corerw(p, buf, addr, r, w)
struct localproc *p;
char *buf;
unsigned long addr;
{
	int fd;

	if (p->starttext <= addr && addr < p->endtext) {
		fd = p->stabfd;
		addr -= p->starttext - p->textoffset;
	}
	else if (p->startdata <= addr && addr < p->enddata) {
		fd = p->corefd;
		addr -= p->startdata - p->dataoffset;
	}
	else if (p->startstack <= addr && addr < p->endstack) {
		fd = p->corefd;
		addr -= p->startstack - p->stackoffset;
	}
	else if (regaddr <= addr && addr < (regaddr+REGSIZE)) {
		addr -= regaddr;
		if (addr == (19 * 4))	{ /* %g0 */
			if (r)
				bzero(buf, 4);
			return 0;
		}
		fd = p->corefd;
		addr += p->regoffset;
	}
	else
		return "corerw:invalid address";
	if (lseek(fd, addr, 0) == -1)
		return "corerw:lseek failed";
	if (w) {
		if (write(fd, buf, w) != w)
			return "corerw:write error";
	} else {
		if (read(fd, buf, r) != r)
			return "corerw:read error";
	}
	return 0;
}
